/*
 * Copyright (c) 2002-2007 TeamDev Ltd. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * The complete licence text can be found at
 * http://www.teamdev.com/winpack/license.jsf
 */
package teamdev.jxcapture.samples.demo;

import org.jdesktop.application.Action;
import org.jdesktop.application.*;

import javax.swing.*;
import java.io.File;

/**
 * @author Ikryanov Vladimir
 */
public class TrayPopupMenu extends JPopupMenu {

    @Resource
    private String[] captureActions;

    private boolean captureOperationEnabled = true;

    private ApplicationSettings settings = ApplicationSettings.getInstance();
    private CaptureOperations operations = CaptureOperations.getInstance();

    public TrayPopupMenu() {
        createGUI();
        configCaptureOperations();
    }

    private void createGUI() {
        ApplicationContext context = Application.getInstance().getContext();
        ResourceMap resourceMap = context.getResourceMap(TrayPopupMenu.class);
        ApplicationActionMap actionMap = context.getActionMap(TrayPopupMenu.class, this);

        captureActions = new String[resourceMap.getInteger("captureActions.size")];

        resourceMap.injectFields(this);
        resourceMap.injectComponents(this);

        JRadioButtonMenuItem toViewerMenuItem = new JRadioButtonMenuItem(actionMap.get("captureToViewer"));
        JRadioButtonMenuItem toFileMenuItem = new JRadioButtonMenuItem(actionMap.get("captureToFile"));
        JRadioButtonMenuItem toClipboardMenuItem = new JRadioButtonMenuItem(actionMap.get("captureToClipboard"));

        toViewerMenuItem.setSelected(settings.isCaptureToViewer());
        toFileMenuItem.setSelected(settings.isCaptureToFile());
        toClipboardMenuItem.setSelected(settings.isCaptureToClipboard());

        ButtonGroup buttonGroup = new ButtonGroup();
        buttonGroup.add(toViewerMenuItem);
        buttonGroup.add(toClipboardMenuItem);
        buttonGroup.add(toFileMenuItem);

        JMenu captureToMenu = new JMenu("Capture To");
        captureToMenu.add(toViewerMenuItem);
        captureToMenu.add(toClipboardMenuItem);
        captureToMenu.add(toFileMenuItem);

        JMenuItem settingsMenuItem = new JMenuItem(actionMap.get("settings"));
        settingsMenuItem.setIcon(resourceMap.getIcon("settings.menu.icon"));
        settingsMenuItem.setAccelerator(null);

        addMenuItems(actionMap, captureActions);
        addSeparator();
        add(captureToMenu);
        addSeparator();
        add(settingsMenuItem);
        addMenuItems(actionMap, new String[]{"about"});
        addSeparator();
        addMenuItems(actionMap, new String[]{"quit"});
    }

    private void configCaptureOperations() {
        operations.addCaptureOperationsListener(new CaptureOperations.CaptureOperationsListener() {
            public void operationBegin() {
                setCaptureOperationEnabled(false);
            }

            public void operationComplete(boolean isCanceled) {
                setCaptureOperationEnabled(true);
            }
        });
    }

    private void addMenuItems(ActionMap actionMap, String[] actionNames) {
        for (String actionName : actionNames) {
            add(new JMenuItem(actionMap.get(actionName)));
        }
    }

    public boolean isCaptureOperationEnabled() {
        return captureOperationEnabled;
    }

    public void setCaptureOperationEnabled(boolean captureOperationEnabled) {
        boolean oldValue = this.captureOperationEnabled;
        this.captureOperationEnabled = captureOperationEnabled;
        firePropertyChange("captureOperationEnabled", oldValue, this.captureOperationEnabled);
    }

    @Action (enabledProperty = "captureOperationEnabled")
    public void activeWindowCapture() {
        operations.activeWindowCapture();
    }

    @Action (enabledProperty = "captureOperationEnabled")
    public void objectCapture() {
        operations.objectCapture();
    }

    @Action (enabledProperty = "captureOperationEnabled")
    public void regionCapture() {
        operations.regionCapture();
    }

    @Action (enabledProperty = "captureOperationEnabled")
    public void desktopCapture() {
        operations.desktopCapture();
    }

    @Action (enabledProperty = "captureOperationEnabled")
    public void openImage() {
        ImageFileChooser fileChooser = new ImageFileChooser();
        File imageFile = fileChooser.openImageFile();
        if (imageFile != null) {
            ImageViewer imageViewer = new ImageViewer();
            if (imageViewer.setImage(imageFile)) {
                imageViewer.setVisible(true);
                imageViewer.toFront();
            }
        }
    }

    @Action
    public void captureToViewer() {
        settings.setCaptureToViewer();
    }

    @Action
    public void captureToFile() {
        settings.setCaptureToFile();
    }

    @Action
    public void captureToClipboard() {
        settings.setCaptureToClipboard();
    }
}